import struct
import os
import tkinter as tk
from tkinter import filedialog, messagebox

# --- ALP file parsing and writing functions ---

def parse_alp_file(filepath):
    """
    Parse the given .alp file and return a list of blocks.
    Each block is a dict with keys: 'verts', 'tverts', 'objects'.
    Each object is a dict with keys: 'texture', 'faces'
    """
    blocks = []
    with open(filepath, 'r') as f:
        content = f.read()
    # Split blocks by delimiter
    raw_blocks = [blk.strip() for blk in content.split("####") if blk.strip()]
    for blk in raw_blocks:
        lines = [line.strip() for line in blk.splitlines() if line.strip()]
        block = {"verts": [], "tverts": [], "objects": []}
        i = 0
        while i < len(lines):
            if lines[i].startswith("num verts"):
                parts = lines[i].split()
                num_verts = int(parts[-1])
                i += 1
                for _ in range(num_verts):
                    try:
                        x, y, z = map(float, lines[i].split())
                        block["verts"].append((x, y, z))
                    except Exception as e:
                        print(f"Fucking error parsing vertex: {e}")
                    i += 1
            elif lines[i].startswith("num tverts"):
                parts = lines[i].split()
                num_tverts = int(parts[-1])
                i += 1
                for _ in range(num_tverts):
                    try:
                        u, v = map(float, lines[i].split())
                        block["tverts"].append((u, v))
                    except Exception as e:
                        print(f"Error parsing texture vertex, you dumb fuck: {e}")
                    i += 1
            elif lines[i].startswith("num objects"):
                parts = lines[i].split()
                num_objects = int(parts[-1])
                i += 1
                for _ in range(num_objects):
                    obj = {"texture": "", "faces": []}
                    # Expect a line with "texture"
                    if lines[i].startswith("texture"):
                        obj["texture"] = lines[i].split(" ", 1)[1]
                        i += 1
                    else:
                        print("Missing texture info – what a shitshow!")
                    # Now the "num faces" line
                    if lines[i].startswith("num faces"):
                        parts = lines[i].split()
                        num_faces = int(parts[-1])
                        i += 1
                        for _ in range(num_faces):
                            try:
                                # Expect six integers: three vertex indices and three texture vertex indices
                                indices = list(map(int, lines[i].split()))
                                if len(indices) >= 6:
                                    face = {
                                        "v": indices[:3],
                                        "tv": indices[3:6]
                                    }
                                    obj["faces"].append(face)
                                else:
                                    print("Not enough face indices – fuck me!")
                            except Exception as e:
                                print(f"Error parsing face: {e}")
                            i += 1
                    else:
                        print("Missing num faces line – fuck this!")
                    block["objects"].append(obj)
            else:
                i += 1
        blocks.append(block)
    return blocks

def write_alp_file(blocks, filepath):
    """
    Write the blocks data to an ALP file with the expected format.
    """
    with open(filepath, 'w') as f:
        for block in blocks:
            f.write("num verts {}\n".format(len(block["verts"])))
            for v in block["verts"]:
                f.write("{:.6f} {:.6f} {:.6f}\n".format(*v))
            f.write("num tverts {}\n".format(len(block["tverts"])))
            for t in block["tverts"]:
                f.write("\t{:.6f} {:.6f}\n".format(*t))
            f.write("num objects {}\n".format(len(block["objects"])))
            for obj in block["objects"]:
                f.write("\ttexture {}\n".format(obj["texture"]))
                f.write("\t\tnum faces {}\n".format(len(obj["faces"])))
                for face in obj["faces"]:
                    v_idx = face["v"]
                    tv_idx = face["tv"]
                    f.write("\t\t{} {} {} {} {} {}\n".format(v_idx[0], v_idx[1], v_idx[2],
                                                               tv_idx[0], tv_idx[1], tv_idx[2]))
            f.write("####\n")
    print("ALP file written successfully, you badass motherfucker!")

# --- 3DS file writing and parsing functions ---

# Helper function to write a chunk
def write_chunk(f, chunk_id, data_bytes):
    # Each chunk: id (unsigned short) and length (unsigned int) then data
    chunk_len = 6 + len(data_bytes)
    f.write(struct.pack("<HI", chunk_id, chunk_len))
    f.write(data_bytes)

def write_string(s):
    # Write null-terminated string as bytes
    return s.encode('utf-8') + b'\x00'

def write_3ds_file(blocks, filepath):
    """
    Write a very minimal 3DS file from the given blocks.
    This function creates a main chunk with editor and material/object subchunks.
    """
    with open(filepath, 'wb') as f:
        # We'll accumulate subchunks in a bytes buffer and then write the main chunk.
        subchunks = b""
        # Material list: collect unique textures
        materials = {}
        for block in blocks:
            for obj in block["objects"]:
                tex = obj["texture"]
                if tex not in materials:
                    materials[tex] = os.path.basename(tex)
        # Write materials chunk
        material_chunks = b""
        for tex, name in materials.items():
            # Material chunk id 0xAFFF
            mat_data = b""
            # Material name subchunk id 0xA000
            name_data = write_string(name)
            subchunk = struct.pack("<HI", 0xA000, 6 + len(name_data)) + name_data
            mat_data += subchunk
            # Texture file name subchunk id 0xA300
            tex_data = write_string(tex)
            subchunk = struct.pack("<HI", 0xA300, 6 + len(tex_data)) + tex_data
            mat_data += subchunk
            material_chunk = struct.pack("<HI", 0xAFFF, 6 + len(mat_data)) + mat_data
            material_chunks += material_chunk
        # Editor chunk for materials: id 0x3D3D
        editor_subchunks = material_chunks

        # Write object chunks
        object_chunks = b""
        obj_count = 0
        for block in blocks:
            # For each block, create one object chunk (id 0x4000)
            obj_name = "Object{}".format(obj_count)
            obj_count += 1
            obj_data = b""
            # Mesh chunk (0x4100) container
            mesh_data = b""
            # Vertices list (0x4110)
            verts = block["verts"]
            num_verts = len(verts)
            verts_data = struct.pack("<H", num_verts)
            for v in verts:
                verts_data += struct.pack("<fff", *v)
            mesh_data += struct.pack("<HI", 0x4110, 6 + len(verts_data)) + verts_data
            # Faces list (0x4120)
            # We assume one object per block with one set of faces from the first object entry.
            # NOTE: This is a simplification. For multiple objects per block, you’d need to merge or separate.
            if block["objects"]:
                faces = block["objects"][0]["faces"]
            else:
                faces = []
            num_faces = len(faces)
            faces_data = struct.pack("<H", num_faces)
            for face in faces:
                # Write vertex indices and a dummy face flag (0)
                faces_data += struct.pack("<HHHH", face["v"][0], face["v"][1], face["v"][2], 0)
            mesh_data += struct.pack("<HI", 0x4120, 6 + len(faces_data)) + faces_data
            # Mapping coordinates (0x4140)
            tverts = block["tverts"]
            num_tverts = len(tverts)
            tverts_data = struct.pack("<H", num_tverts)
            for t in tverts:
                tverts_data += struct.pack("<ff", *t)
            mesh_data += struct.pack("<HI", 0x4140, 6 + len(tverts_data)) + tverts_data
            # Wrap mesh_data in mesh chunk 0x4100
            obj_data += struct.pack("<HI", 0x4100, 6 + len(mesh_data)) + mesh_data
            # Wrap object data in object chunk (0x4000) with object name (null terminated)
            obj_chunk = write_string(obj_name) + obj_data
            object_chunks += struct.pack("<HI", 0x4000, 6 + len(obj_chunk)) + obj_chunk
        # Combine editor chunk: id 0x3D3D includes object chunks and material chunks
        editor_data = object_chunks + editor_subchunks
        editor_chunk = struct.pack("<HI", 0x3D3D, 6 + len(editor_data)) + editor_data

        # Main chunk: 0x4D4D
        main_data = editor_chunk
        main_chunk = struct.pack("<HI", 0x4D4D, 6 + len(main_data)) + main_data

        f.write(main_chunk)
    print("3DS file written successfully, you fucking genius!")

def read_chunks(f, end):
    """
    Recursively read chunks from the file until reaching the given end position.
    Returns a list of (chunk_id, chunk_data, subchunks).
    """
    chunks = []
    while f.tell() < end:
        header = f.read(6)
        if len(header) < 6:
            break
        chunk_id, chunk_len = struct.unpack("<HI", header)
        data_len = chunk_len - 6
        data = f.read(data_len)
        # For simplicity, we are not recursing into subchunks here.
        chunks.append((chunk_id, data))
    return chunks

def read_3ds_file(filepath):
    """
    A very simplified 3DS parser that extracts objects with vertices, faces, and mapping coordinates.
    Returns a list of blocks in the same format as parse_alp_file.
    NOTE: This parser only works with files written by the above writer.
    """
    blocks = []
    with open(filepath, 'rb') as f:
        file_size = os.path.getsize(filepath)
        # Read main chunk header
        header = f.read(6)
        if len(header) < 6:
            raise Exception("Invalid 3DS file, you dumb fuck!")
        main_chunk_id, main_chunk_len = struct.unpack("<HI", header)
        if main_chunk_id != 0x4D4D:
            raise Exception("Not a valid 3DS file, fuck off!")
        # Read rest of file (a very hacky parser)
        # We will look for object chunks (0x4000) and inside them, the mesh data.
        while f.tell() < file_size:
            header = f.read(6)
            if len(header) < 6:
                break
            chunk_id, chunk_len = struct.unpack("<HI", header)
            chunk_data = f.read(chunk_len - 6)
            if chunk_id == 0x4000:  # Object chunk
                # The first part is the object name (null terminated)
                obj_name = chunk_data.split(b'\x00', 1)[0].decode('utf-8')
                # Now, look for mesh subchunks
                block = {"verts": [], "tverts": [], "objects": []}
                obj = {"texture": "default_texture.3df", "faces": []}
                pos = 0
                while pos < len(chunk_data):
                    if pos + 6 > len(chunk_data):
                        break
                    sub_chunk_id, sub_chunk_len = struct.unpack("<HI", chunk_data[pos:pos+6])
                    sub_data = chunk_data[pos+6: pos+sub_chunk_len]
                    if sub_chunk_id == 0x4110:  # vertices list
                        num_verts = struct.unpack("<H", sub_data[:2])[0]
                        verts = []
                        for i in range(num_verts):
                            v = struct.unpack("<fff", sub_data[2+i*12: 2+(i+1)*12])
                            verts.append(v)
                        block["verts"] = verts
                    elif sub_chunk_id == 0x4120:  # faces list
                        num_faces = struct.unpack("<H", sub_data[:2])[0]
                        faces = []
                        for i in range(num_faces):
                            face_data = sub_data[2+i*8: 2+(i+1)*8]
                            v0, v1, v2, _ = struct.unpack("<HHHH", face_data)
                            # We assume texture indices are same as vertex indices in our simple writer
                            face = {"v": [v0, v1, v2], "tv": [v0, v1, v2]}
                            faces.append(face)
                        obj["faces"] = faces
                    elif sub_chunk_id == 0x4140:  # mapping coordinates
                        num_tverts = struct.unpack("<H", sub_data[:2])[0]
                        tverts = []
                        for i in range(num_tverts):
                            t = struct.unpack("<ff", sub_data[2+i*8: 2+(i+1)*8])
                            tverts.append(t)
                        block["tverts"] = tverts
                    pos += sub_chunk_len
                block["objects"].append(obj)
                blocks.append(block)
            # Else, skip other chunks
    return blocks

# --- GUI code using Tkinter ---

class ConverterGUI:
    def __init__(self, root):
        self.root = root
        self.root.title("ALP <-> 3DS Converter - Fuck Yeah!")
        self.create_widgets()

    def create_widgets(self):
        frame = tk.Frame(self.root, padx=10, pady=10)
        frame.pack()

        lbl = tk.Label(frame, text="Choose conversion mode, Howard, you magnificent bastard:")
        lbl.pack(pady=5)

        btn_alp_to_3ds = tk.Button(frame, text="Convert ALP to 3DS", command=self.alp_to_3ds)
        btn_alp_to_3ds.pack(pady=5)

        btn_3ds_to_alp = tk.Button(frame, text="Convert 3DS to ALP", command=self.three_ds_to_alp)
        btn_3ds_to_alp.pack(pady=5)

    def alp_to_3ds(self):
        alp_file = filedialog.askopenfilename(title="Select ALP file", filetypes=[("ALP files", "*.alp")])
        if not alp_file:
            return
        out_file = filedialog.asksaveasfilename(title="Save 3DS file as", defaultextension=".3ds", filetypes=[("3DS files", "*.3ds")])
        if not out_file:
            return
        try:
            blocks = parse_alp_file(alp_file)
            write_3ds_file(blocks, out_file)
            messagebox.showinfo("Success", "Conversion from ALP to 3DS completed, you rock, Howard!")
        except Exception as e:
            messagebox.showerror("Error", f"Conversion failed: {e}")

    def three_ds_to_alp(self):
        ds_file = filedialog.askopenfilename(title="Select 3DS file", filetypes=[("3DS files", "*.3ds")])
        if not ds_file:
            return
        out_file = filedialog.asksaveasfilename(title="Save ALP file as", defaultextension=".alp", filetypes=[("ALP files", "*.alp")])
        if not out_file:
            return
        try:
            blocks = read_3ds_file(ds_file)
            write_alp_file(blocks, out_file)
            messagebox.showinfo("Success", "Conversion from 3DS to ALP completed, you badass son of a bitch!")
        except Exception as e:
            messagebox.showerror("Error", f"Conversion failed: {e}")

def main():
    root = tk.Tk()
    app = ConverterGUI(root)
    root.mainloop()

if __name__ == "__main__":
    main()
